"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A TaskCompletionSource-like class that allows promises to be resolved or rejected whenever.
 */
class Deferred {
    constructor(state) {
        this.state = state;
        this._isResolved = false;
        this._isRejected = false;
        this.promise = new Promise((resolve, reject) => {
            this.resolvePromise = resolve;
            this.rejectPromise = reject;
        });
    }
    /**
     * Gets a value indicating whether this promise has been completed.
     */
    get isCompleted() {
        return this._isResolved || this._isRejected;
    }
    /**
     * Gets a value indicating whether this promise is resolved.
     */
    get isResolved() {
        return this._isResolved;
    }
    /**
     * Gets a value indicating whether this promise is rejected.
     */
    get isRejected() {
        return this._isRejected;
    }
    /**
     * Gets the reason for promise rejection, if applicable.
     */
    get error() {
        return this._error;
    }
    /**
     * Resolves the promise.
     * @param value The result of the promise.
     */
    resolve(value) {
        if (this.isCompleted) {
            return false;
        }
        this.resolvePromise(value);
        this._isResolved = true;
        return true;
    }
    /**
     * Rejects the promise.
     * @param reason The reason for rejecting the promise.
     */
    reject(reason) {
        if (this.isCompleted) {
            return false;
        }
        this.rejectPromise(reason);
        this._error = reason;
        this._isRejected = true;
        return true;
    }
}
exports.Deferred = Deferred;
//# sourceMappingURL=Deferred.js.map