"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const main_1 = require("../main");
const fs = require("fs");
const path = require("path");
const util = require("util");
const process = require("process");
const commander_1 = require("commander");
const Tracer = require("pegjs-backtrace");
function deleteLocation(node) {
    if (!node) {
        return;
    }
    if (node.hasOwnProperty('location')) {
        delete node.location;
    }
    for (const key of Object.getOwnPropertyNames(node)) {
        const val = node[key];
        if (typeof val === 'object') {
            deleteLocation(val);
        }
    }
}
commander_1.program
    .option('-i, --inspect', 'use util.inspect to output AST')
    .option('--color', 'turn on the color option of util.inspect')
    .option('-l, --location', 'enable location')
    .option('-c, --comment', 'enable comment')
    .option('-s, --start-rule [rule]', 'set start rule. default is "Root".')
    .option('-d, --debug', 'enable backtrace for debug')
    .option('--timeout [timeout]', 'set tiemout in milliseconds')
    .parse(process.argv);
const filename = commander_1.program.args[0];
if (!fs.existsSync(filename)) {
    console.error(`${filename} not found.`);
    process.exit(1);
}
const opts = commander_1.program.opts();
const s = fs.readFileSync(filename, { encoding: 'utf8' });
const startRule = opts.startRule || 'Root';
const ext = path.extname(filename);
let ret;
const useColor = opts.color ? true : false;
const tracer = opts.debug ? new Tracer(s, { showTrace: true, useColor, }) : undefined;
const timeout = Number(opts.timeout);
try {
    if (ext === '.tex') {
        ret = main_1.latexParser.parse(s, { startRule, enableComment: opts.comment, tracer, timeout });
        if (!opts.location) {
            deleteLocation(ret);
        }
    }
    else if (ext === '.bib') {
        ret = main_1.bibtexParser.parse(s, { tracer });
    }
    else if (ext === '.log') {
        ret = main_1.latexLogParser.parse(s, { tracer });
    }
    else {
        console.error('The suffix of the file is unknown.');
        process.exit(1);
    }
}
catch (e) {
    if (main_1.latexParser.isSyntaxError(e)) {
        if (tracer) {
            console.log(tracer.getBacktraceString());
        }
        const loc = e.location;
        console.error(`SyntaxError at line: ${loc.start.line}, column: ${loc.start.column}.`);
        console.error(e.message);
        process.exit(1);
    }
    throw e;
}
if (opts.inspect) {
    const colors = opts.color;
    console.log(util.inspect(ret, { showHidden: false, depth: null, colors }));
}
else {
    console.log(JSON.stringify(ret, undefined, '  '));
}
//# sourceMappingURL=luparse.js.map