"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonRpcConnection = exports.ServiceJsonRpcDescriptor = void 0;
const vscode_jsonrpc_1 = require("vscode-jsonrpc");
const constants_1 = require("./constants");
const ServiceMoniker_1 = require("./ServiceMoniker");
const ServiceRpcDescriptor_1 = require("./ServiceRpcDescriptor");
const utilities_1 = require("./utilities");
const BigEndianInt32LengthHeaderMessageHandler_1 = require("./BigEndianInt32LengthHeaderMessageHandler");
const msgpack = require("msgpack-lite");
/**
 * Constructs a JSON RPC message connection to a service
 */
class ServiceJsonRpcDescriptor extends ServiceRpcDescriptor_1.ServiceRpcDescriptor {
    /**
     * Initializes a new instance of the [ServiceJsonRpcDescriptor](#ServiceJsonRpcDescriptor) class
     * @param moniker The moniker this descriptor describes
     * @param formatter The formatter to use when sending messages
     * @param messageDelimiter The delimiter to use in separating messages
     */
    constructor(moniker, formatter, messageDelimiter) {
        super(moniker);
        this.formatter = formatter;
        this.messageDelimiter = messageDelimiter;
        this.protocol = 'json-rpc';
        let contentTypeEncoder;
        let contentTypeDecoder;
        switch (formatter) {
            case constants_1.Formatters.Utf8:
                contentTypeEncoder = msg => Promise.resolve(Buffer.from(JSON.stringify(msg)));
                contentTypeDecoder = buffer => JSON.parse(buffer.toString());
                break;
            case constants_1.Formatters.MessagePack:
                contentTypeEncoder = msg => Promise.resolve(msgpack.encode(msg));
                contentTypeDecoder = buffer => msgpack.decode(buffer);
                break;
            default:
                throw new Error(`Unsupported formatter: ${formatter}.`);
        }
        if (messageDelimiter === constants_1.MessageDelimiters.HttpLikeHeaders) {
            if (formatter !== constants_1.Formatters.Utf8) {
                // The limited configurations we can support here can be relaxed after we update to vscode-jsonrpc 6.0,
                // which lets us mix-and-match the content type encoder (handler/delimiter) and the content encoder (foramtter) independently.
                throw new Error(`Utf8 is the only formatter supported while using HttpLikeHeaders.`);
            }
            this.connectionFactory = rw => vscode_jsonrpc_1.createMessageConnection(rw, rw);
        }
        else if (messageDelimiter === constants_1.MessageDelimiters.BigEndianInt32LengthHeader) {
            this.connectionFactory = rw => vscode_jsonrpc_1.createMessageConnection(new BigEndianInt32LengthHeaderMessageHandler_1.BE32MessageReader(rw, contentTypeDecoder), new BigEndianInt32LengthHeaderMessageHandler_1.BE32MessageWriter(rw, contentTypeEncoder));
        }
        else {
            throw new Error(`Unsupported message delimiter: ${messageDelimiter}.`);
        }
    }
    constructRpcConnection(pipe) {
        return new JsonRpcConnection(utilities_1.constructMessageConnection(pipe, this.connectionFactory));
    }
    equals(descriptor) {
        if (!descriptor || !(descriptor instanceof ServiceJsonRpcDescriptor)) {
            return false;
        }
        if (!ServiceMoniker_1.ServiceMoniker.equals(descriptor.moniker, this.moniker)) {
            return false;
        }
        return (this.formatter === descriptor.formatter && this.messageDelimiter === descriptor.messageDelimiter);
    }
}
exports.ServiceJsonRpcDescriptor = ServiceJsonRpcDescriptor;
const rpcProxy = {
    get: (target, property) => {
        switch (property.toString()) {
            case "dispose":
                return function () {
                    target.dispose();
                };
            case "then":
                // When the proxy is returned from async methods,
                // promises look at the return value to see if it too is a promise.
                return undefined;
            default:
                return function () {
                    const methodName = property.toString();
                    // vscode-jsonrpc behaves erratically unless all args are passed together in an array, with any CancellationToken separated out.
                    // See https://github.com/microsoft/vscode-languageserver-node/issues/630
                    let ct;
                    if (arguments.length > 0) {
                        if (vscode_jsonrpc_1.CancellationToken.is(arguments[arguments.length - 1])) {
                            ct = arguments[arguments.length - 1];
                            const args = validateNoUndefinedElements(Array.prototype.slice.call(arguments, 0, arguments.length - 1));
                            return target.sendRequest(methodName, args, ct);
                        }
                        else if (arguments[arguments.length - 1] === undefined) {
                            // The last arg is most likely a `CancellationToken?` that was propagated to the RPC call from another method that made it optional.
                            // We can't tell, but we mustn't claim it's a CancellationToken nor an ordinary argument or else an RPC server
                            // may fail to match the RPC call to a method because of an extra argument.
                            // If this truly was a value intended to propagate, they should use `null` as the argument.
                            const args = validateNoUndefinedElements(Array.prototype.slice.call(arguments, 0, arguments.length - 1));
                            return target.sendRequest(methodName, args);
                        }
                    }
                    const validatedArgs = validateNoUndefinedElements(Array.prototype.slice.call(arguments));
                    return target.sendRequest(methodName, validatedArgs);
                };
        }
    },
};
function validateNoUndefinedElements(values) {
    for (let i = 0; i < values.length; i++) {
        if (values[i] === undefined) {
            throw new Error(`Argument at 0-based index ${i} is set as undefined, which is not a valid JSON value.`);
        }
    }
    return values;
}
class JsonRpcConnection extends ServiceRpcDescriptor_1.RpcConnection {
    constructor(messageConnection) {
        super();
        this.messageConnection = messageConnection;
    }
    addLocalRpcTarget(rpcTarget) {
        function registerRequestAndNotification(connection, methodName, method) {
            connection.onRequest(methodName, (...args) => method.apply(rpcTarget, args));
            connection.onNotification(methodName, (...args) => method.apply(rpcTarget, args));
        }
        JsonRpcConnection.getInstanceMethodNames(rpcTarget, Object.prototype).forEach(methodName => {
            if (methodName !== 'dispose') {
                const method = rpcTarget[methodName];
                registerRequestAndNotification(this.messageConnection, methodName, method);
                // Add an alias for the method so that we support with and without the Async suffix.
                const suffix = 'Async';
                const alias = methodName.endsWith(suffix) ? methodName.substring(0, methodName.length - suffix.length) : `${methodName}${suffix}`;
                registerRequestAndNotification(this.messageConnection, alias, method);
            }
        });
        if (typeof rpcTarget.dispose === 'function') {
            this.messageConnection.onDispose(() => rpcTarget.dispose());
        }
    }
    constructRpcClient() {
        return new Proxy(this.messageConnection, rpcProxy);
    }
    startListening() {
        var _a;
        (_a = this.messageConnection) === null || _a === void 0 ? void 0 : _a.listen();
    }
    dispose() {
    }
    static isMethod(obj, name) {
        const desc = Object.getOwnPropertyDescriptor(obj, name);
        return !!desc && typeof desc.value === 'function';
    }
    static getInstanceMethodNames(obj, stopPrototype) {
        const array = [];
        let proto = Object.getPrototypeOf(obj);
        while (proto && proto !== stopPrototype) {
            Object.getOwnPropertyNames(proto)
                .forEach(name => {
                if (name !== 'constructor') {
                    if (JsonRpcConnection.isMethod(proto, name)) {
                        array.push(name);
                    }
                }
            });
            proto = Object.getPrototypeOf(proto);
        }
        return array;
    }
}
exports.JsonRpcConnection = JsonRpcConnection;

// SIG // Begin signature block
// SIG // MIIhgQYJKoZIhvcNAQcCoIIhcjCCIW4CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // Cx2WyUYXA+H/SN3j2XvoXP/4mWbNy32Pz+AUH3Rp9+Kg
// SIG // ggtyMIIE+jCCA+KgAwIBAgITMwAAAyVIsp0Of8X0HwAA
// SIG // AAADJTANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTIwMDMwNDE4MjkyOVoX
// SIG // DTIxMDMwMzE4MjkyOVowdDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEeMBwGA1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // o6USNjR5nReBPqM/t9oEDRVn4lgP4FtusaT6n9JeO7ff
// SIG // aeyCCneQrkwDHjzWFlx0k/KRX2/MmkzVs+pxAmm4UEwi
// SIG // MQKrQM1aQjVl4rkA3tfHhFKFBTdrIKDIaVO4G9IGYoG4
// SIG // b855k9JqKu6KwRADHK1ugvPPvScqlgYAytDAW6cYyMAb
// SIG // 9Lj4I90eNCLTC0MPS29EgZ64jhD0ChgwjtgCOjEovvsX
// SIG // uWFzpEGaywg3Ok5f3/30859QoviUe2CTHXavH9jXuNqI
// SIG // LFFSFuiR/Eimy0F2/cTlZ+HzIhvW+j4EboAKUtCCgFCP
// SIG // DkENpEq9mtQbChGPQezDVHhQVwe+Dq7rWQIDAQABo4IB
// SIG // eTCCAXUwHwYDVR0lBBgwFgYKKwYBBAGCNz0GAQYIKwYB
// SIG // BQUHAwMwHQYDVR0OBBYEFC3CjwCdK+IVK01UmJqWuUQP
// SIG // rh58MFAGA1UdEQRJMEekRTBDMSkwJwYDVQQLEyBNaWNy
// SIG // b3NvZnQgT3BlcmF0aW9ucyBQdWVydG8gUmljbzEWMBQG
// SIG // A1UEBRMNMjMwODY1KzQ1ODQ5MzAfBgNVHSMEGDAWgBTm
// SIG // /F97uyIAWORyTrX0IXQjMubvrDBWBgNVHR8ETzBNMEug
// SIG // SaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtp
// SIG // L2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0FfMjAxMC0w
// SIG // Ny0wNi5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
// SIG // BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
// SIG // L2NlcnRzL01pY0NvZFNpZ1BDQV8yMDEwLTA3LTA2LmNy
// SIG // dDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEBCwUAA4IB
// SIG // AQBcc+DvnFngaPOnuRCJolHiV6CeYbicB+pd2FAS4Qbf
// SIG // rfrhOsw7QapsoHgKvqjLN6LWrnE5julRMi0QAmu2MfMc
// SIG // QyyuzAHJ8cAGOkmI3IhKzbA39bQKXmZAiCDRVv/EtkCj
// SIG // EkF3+DiaJY1h6NU5J4i/SEZs74jfZeNjWDSQfYu9Xs8T
// SIG // /umTeDw1ovEFXu9HRgi5fTtkcT0BYRHJd41nQruQtRQ0
// SIG // YOqjKXMd6rHkEdkBePneBBP1ENYd2HkoW7bs8X9JLGwZ
// SIG // 0KlVNmSfRPxiVE1lErGMbCZxvMYNxuOic97bX1RN3KDC
// SIG // yodXx6rCN11yJVc09wSMdzfZy4NfIcDjbbuZMIIGcDCC
// SIG // BFigAwIBAgIKYQxSTAAAAAAAAzANBgkqhkiG9w0BAQsF
// SIG // ADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
// SIG // bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
// SIG // FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMp
// SIG // TWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9y
// SIG // aXR5IDIwMTAwHhcNMTAwNzA2MjA0MDE3WhcNMjUwNzA2
// SIG // MjA1MDE3WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
// SIG // V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
// SIG // A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYD
// SIG // VQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAy
// SIG // MDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEA6Q5kUHlntcTj/QkATJ6UrPdWaOpE2M/FWE+ppXZ8
// SIG // bUW60zmStKQe+fllguQX0o/9RJwI6GWTzixVhL99COMu
// SIG // K6hBKxi3oktuSUxrFQfe0dLCiR5xlM21f0u0rwjYzIjW
// SIG // axeUOpPOJj/s5v40mFfVHV1J9rIqLtWFu1k/+JC0K4N0
// SIG // yiuzO0bj8EZJwRdmVMkcvR3EVWJXcvhnuSUgNN5dpqWV
// SIG // XqsogM3Vsp7lA7Vj07IUyMHIiiYKWX8H7P8O7YASNUwS
// SIG // pr5SW/Wm2uCLC0h31oVH1RC5xuiq7otqLQVcYMa0Kluc
// SIG // IxxfReMaFB5vN8sZM4BqiU2jamZjeJPVMM+VHwIDAQAB
// SIG // o4IB4zCCAd8wEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0O
// SIG // BBYEFOb8X3u7IgBY5HJOtfQhdCMy5u+sMBkGCSsGAQQB
// SIG // gjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAP
// SIG // BgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP
// SIG // 6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeG
// SIG // RWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3Js
// SIG // L3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIz
// SIG // LmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKG
// SIG // Pmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2Vy
// SIG // dHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MIGd
// SIG // BgNVHSAEgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9Bggr
// SIG // BgEFBQcCARYxaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
// SIG // L1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0bTBABggrBgEF
// SIG // BQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABpAGMA
// SIG // eQBfAFMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG
// SIG // 9w0BAQsFAAOCAgEAGnTvV08pe8QWhXi4UNMi/AmdrIKX
// SIG // +DT/KiyXlRLl5L/Pv5PI4zSp24G43B4AvtI1b6/lf3mV
// SIG // d+UC1PHr2M1OHhthosJaIxrwjKhiUUVnCOM/PB6T+DCF
// SIG // F8g5QKbXDrMhKeWloWmMIpPMdJjnoUdD8lOswA8waX/+
// SIG // 0iUgbW9h098H1dlyACxphnY9UdumOUjJN2FtB91TGcun
// SIG // 1mHCv+KDqw/ga5uV1n0oUbCJSlGkmmzItx9KGg5pqdfc
// SIG // wX7RSXCqtq27ckdjF/qm1qKmhuyoEESbY7ayaYkGx0aG
// SIG // ehg/6MUdIdV7+QIjLcVBy78dTMgW77Gcf/wiS0mKbhXj
// SIG // pn92W9FTeZGFndXS2z1zNfM8rlSyUkdqwKoTldKOEdqZ
// SIG // Z14yjPs3hdHcdYWch8ZaV4XCv90Nj4ybLeu07s8n07Ve
// SIG // afqkFgQBpyRnc89NT7beBVaXevfpUk30dwVPhcbYC/GO
// SIG // 7UIJ0Q124yNWeCImNr7KsYxuqh3khdpHM2KPpMmRM19x
// SIG // HkCvmGXJIuhCISWKHC1g2TeJQYkqFg/XYTyUaGBS79ZH
// SIG // maCAQO4VgXc+nOBTGBpQHTiVmx5mMxMnORd4hzbOTsNf
// SIG // svU9R1O24OXbC2E9KteSLM43Wj5AQjGkHxAIwlacvyRd
// SIG // UQKdannSF9PawZSOB3slcUSrBmrm1MbfI5qWdcUxghVn
// SIG // MIIVYwIBATCBlTB+MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgw
// SIG // JgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBD
// SIG // QSAyMDEwAhMzAAADJUiynQ5/xfQfAAAAAAMlMA0GCWCG
// SIG // SAFlAwQCAQUAoIGuMBkGCSqGSIb3DQEJAzEMBgorBgEE
// SIG // AYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3
// SIG // AgEVMC8GCSqGSIb3DQEJBDEiBCA0vs923jKG6DhUnjkr
// SIG // Fvf5FNEMFQgDS1xYEDsXKP8vaDBCBgorBgEEAYI3AgEM
// SIG // MTQwMqAUgBIATQBpAGMAcgBvAHMAbwBmAHShGoAYaHR0
// SIG // cDovL3d3dy5taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEB
// SIG // AQUABIIBAJwEzdPnmQLTcs6WtYxw2SaIP/e7kckCwYjW
// SIG // nTQsX9Zv6RghYnFrlbVYudj3Z8wvbK5oITkhr0ftd0rW
// SIG // I3SgoGd/XwmglnWLpk9rPGvUci4/ctJfVI5sDfgvethi
// SIG // Rfw5f5lkFfpslLCiEMiXsckanZY4NWPTj+rmlpnFUVll
// SIG // Z8+89YUHT9HFbKSKL7ci6ABgPCDhQwq9+hUOZ/1YhqrD
// SIG // OedbTRHEwAspbiLxwOK8K9ejkfhZv4o8EwQ1UMdT3KaG
// SIG // TD+wPE3pDpEGyFBPdQCmJXrDamNg9Vaby7f21fylO5OE
// SIG // vUw3sHE/Y0R1MId6rB62YRtV3VdORMT/1NWExSWBo+eh
// SIG // ghLxMIIS7QYKKwYBBAGCNwMDATGCEt0wghLZBgkqhkiG
// SIG // 9w0BBwKgghLKMIISxgIBAzEPMA0GCWCGSAFlAwQCAQUA
// SIG // MIIBVQYLKoZIhvcNAQkQAQSgggFEBIIBQDCCATwCAQEG
// SIG // CisGAQQBhFkKAwEwMTANBglghkgBZQMEAgEFAAQgEKkQ
// SIG // J7fuA2oxN6Iury7JCug4J7tdgMNJgcMCV2NPLUICBl7V
// SIG // Oj0ZthgTMjAyMDA2MTExODQ1MjQuODM3WjAEgAIB9KCB
// SIG // 1KSB0TCBzjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
// SIG // c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
// SIG // BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEpMCcGA1UE
// SIG // CxMgTWljcm9zb2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJp
// SIG // Y28xJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOkM0QkQt
// SIG // RTM3Ri01RkZDMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBTZXJ2aWNloIIORDCCBPUwggPdoAMCAQIC
// SIG // EzMAAAEjOLDkrdhakJ0AAAAAASMwDQYJKoZIhvcNAQEL
// SIG // BQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
// SIG // bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
// SIG // FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcN
// SIG // MTkxMjE5MDExNDU2WhcNMjEwMzE3MDExNDU2WjCBzjEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEpMCcGA1UECxMgTWljcm9z
// SIG // b2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJpY28xJjAkBgNV
// SIG // BAsTHVRoYWxlcyBUU1MgRVNOOkM0QkQtRTM3Ri01RkZD
// SIG // MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
// SIG // ZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAnbzQybBkpdwBLvHZm8DhM44LPD7rdez1QsZa
// SIG // 11kM3dWX5oZASwzASsiSDNCLR9M7Sw4P03eE7UdpNYeh
// SIG // LzQ39BvqgtHZYJmS/9UzhYWdOE6/fIDnNK36+4o3CuMQ
// SIG // cULSOUwMImppTtK3pYluX+QA/myAzSq2kQRCHG1Vp/wi
// SIG // hXmWry+Awk2vfQ7iuotgSL9hlZBljBAcCJUy6cJikmJx
// SIG // yc041FF2DYPdPK7bZ4QnA9A/oOR4SKgzL16EyYGuSMAN
// SIG // U6BBX5PiaKv6EAl4g3KymzrCBE7mqO5Xn6O9zM1BrVab
// SIG // uPGyoG/TgYKUink0e+tdCZn2all2PeuPEW5lsqN3cQID
// SIG // AQABo4IBGzCCARcwHQYDVR0OBBYEFIHUbBSA040b+RHC
// SIG // sGjeGRX4DJ4eMB8GA1UdIwQYMBaAFNVjOlyKMZDzQ3t8
// SIG // RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6
// SIG // Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1
// SIG // Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNybDBa
// SIG // BggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6
// SIG // Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWlj
// SIG // VGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJKoZI
// SIG // hvcNAQELBQADggEBAFvlAbeqV+hbqvVXiVP6Q7wtTMXf
// SIG // ZLd9R+Cf9LVBAE/M5Gz/q6OPT3K0dY0N857DCRLJrV/x
// SIG // L174FudeScfmXdHqdLYGRFMA21OZfG8wtMLK95h78lAh
// SIG // +iz5neInRvWocNKcSPpCZ1/UzKas8CTmPGHGGKJeXAgt
// SIG // SO8fnrLussfErTCewfXYQ70yeRpI1ck0KZKZ+BQSQM3O
// SIG // 7ncLf2Xpc1EA9q7Pb9ayUhRlxfc0MIyC/mFmLaeF330f
// SIG // HJokmxyfV/yFlcD75/Uc1urxt2SHc5iBGc2vtB2c74a6
// SIG // +27d3Iaph1AwwY+cC3gvsTD3KSPLRSjPrj+vRJtAhFi3
// SIG // Ll4z0zcwggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0G
// SIG // CSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZp
// SIG // Y2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEyMTM2
// SIG // NTVaFw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFBDQSAyMDEwMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEAqR0NvHcRijog7PwTl/X6f2mUa3RU
// SIG // ENWlCgCChfvtfGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9B
// SIG // GxqVHc4JE458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs
// SIG // 1nMwVyaCo0UN0Or1R4HNvyRgMlhgRvJYR4YyhB50YWeR
// SIG // X4FUsc+TTJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcS
// SIG // chohiq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/
// SIG // 9WbAA5ZEfu/QS/1u5ZrKsajyeioKMfDaTgaRtogINeh4
// SIG // HLDpmc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwB
// SIG // BU8iTQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUBBAMC
// SIG // AQAwHQYDVR0OBBYEFNVjOlyKMZDzQ3t8RhvFM2hahW1V
// SIG // MBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
// SIG // DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQY
// SIG // MBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRP
// SIG // ME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNv
// SIG // bS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8y
// SIG // MDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQROMEwwSgYI
// SIG // KwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNv
// SIG // bS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
// SIG // MjMuY3J0MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQB
// SIG // gjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0
// SIG // Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwA
// SIG // XwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBtAGUAbgB0
// SIG // AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCY
// SIG // P4FxAz2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LFZslq
// SIG // 3/Xn8Hi9x6ieJeP5vO1rVFcIK1GCRBL7uVOMzPRgEop2
// SIG // zEBAQZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3fVo/
// SIG // HPKZeUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6AG9LMEQk
// SIG // IjzP7QOllo9ZKby2/QThcJ8ySif9Va8v/rbljjO7Yl+a
// SIG // 21dA6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFjnXshbcOc
// SIG // o6I8+n99lmqQeKZt0uGc+R38ONiU9MalCpaGpL2eGq4E
// SIG // QoO4tYCbIjggtSXlZOz39L9+Y1klD3ouOVd2onGqBooP
// SIG // iRa6YacRy5rYDkeagMXQzafQ732D8OE7cQnfXXSYIghh
// SIG // 2rBQHm+98eEA3+cxB6STOvdlR3jo+KhIq/fecn5ha293
// SIG // qYHLpwmsObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rI
// SIG // DVWZeodzOwjmmC3qjeAzLhIp9cAvVCch98isTtoouLGp
// SIG // 25ayp0Kiyc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8b
// SIG // xyGvWbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/
// SIG // XU/pnR4ZOC+8z1gFLu8NoFA12u8JJxzVs341Hgi62jbb
// SIG // 01+P3nSISRKhggLSMIICOwIBATCB/KGB1KSB0TCBzjEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEpMCcGA1UECxMgTWljcm9z
// SIG // b2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJpY28xJjAkBgNV
// SIG // BAsTHVRoYWxlcyBUU1MgRVNOOkM0QkQtRTM3Ri01RkZD
// SIG // MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
// SIG // ZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQC6F2aN4OKeF8Lu
// SIG // DDUoEJ4z+/tXgaCBgzCBgKR+MHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFBDQSAyMDEwMA0GCSqGSIb3DQEBBQUAAgUA4ozn
// SIG // IjAiGA8yMDIwMDYxMTIxMjM0NloYDzIwMjAwNjEyMjEy
// SIG // MzQ2WjB3MD0GCisGAQQBhFkKBAExLzAtMAoCBQDijOci
// SIG // AgEAMAoCAQACAhhqAgH/MAcCAQACAhHJMAoCBQDijjii
// SIG // AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
// SIG // AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZI
// SIG // hvcNAQEFBQADgYEAPyGbhEao5b49M+0tBsLXfOveMEXV
// SIG // 3m3JEvTs4lrt2XdWevdxkHXcsDx8h8fwqVdYwLXLyD9M
// SIG // 024RqV65+ap7g4aWayvnqjcGXCyWGxTn+KlG/zof5SeD
// SIG // BS32X9//z9pN94/i2IMpa7yPEvI/OevcUS/dgVPe4Jsj
// SIG // ifUFiF0Y4UcxggMNMIIDCQIBATCBkzB8MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0EgMjAxMAITMwAAASM4sOSt2FqQnQAA
// SIG // AAABIzANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcN
// SIG // AQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEi
// SIG // BCCznzvQjkItj9u5bT0Rqqgs/fBxR9XUXo+nrdenkY1q
// SIG // WDCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIBGa
// SIG // M4M/+0TMxA2jo6zEpAAMvynAomQzlidcqur7FYGzMIGY
// SIG // MIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
// SIG // c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
// SIG // BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UE
// SIG // AxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAC
// SIG // EzMAAAEjOLDkrdhakJ0AAAAAASMwIgQgHlAdEi7YaHWm
// SIG // 5eWmGRohznjv9+YiWKzkfhm9AH/tKZ0wDQYJKoZIhvcN
// SIG // AQELBQAEggEAf0mL+4/tztW+qm+QSPHNSuDM2ZvOBEiR
// SIG // Ivzw40D1JeXMPhlcaPcajGlaNaRD8SJRGQAKSy8FFRB3
// SIG // PaNeB0o/4xFlDzsflVGOqoChC8mvKXtxKqxmdZuqRRAo
// SIG // 5SME9YEkymQ2Ax5qELAUaRR6jSplL2v/ktzEh2mkt/r+
// SIG // 4Ax5o+4O3TEofmITN7I1s3iuY33hz3l8QMIp0nri5l+E
// SIG // sXw2V1JlXsBwb/z/M2pB0LTMjHmAQ+nSmaqCQk1qBfVF
// SIG // c+d2OBjuACt3Y2IaVbT0VdQOJ0h2jL4ZM/Mz1LcbpTAH
// SIG // uWLPJTFx6zOddultKPQelAqsSNQr09HZ0QjKWfJLZfxkNw==
// SIG // End signature block
