import torch
import torch.nn.functional as F
import numpy as np
import imageio
import math
import cv2

# -- load EXR HDR environment map as [1,3,H_e,W_e] tensor --
def load_exr_envmap(path, device='cpu'):
    img = cv2.imread(path, cv2.IMREAD_UNCHANGED)
    if img is None:
        raise FileNotFoundError(f"Could not load EXR file: {path}")
    if img.shape[2] == 4:
        img = img[..., :3]  # Drop alpha if present
    img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
    env = torch.from_numpy(img).permute(2,0,1).unsqueeze(0).float().to(device)
    return env  # [1,3,H,W]

# -- precompute SH moments (c0, c1) of the env map --
def compute_sh_coeffs(env):
    # env: [1,3,H_e,W_e]
    _,C,H_e,W_e = env.shape
    device = env.device

    # build direction for each pixel
    i = torch.arange(W_e, device=device).float() + 0.5
    j = torch.arange(H_e, device=device).float() + 0.5
    phi   = i / W_e * 2 * math.pi      # [W_e]
    theta = j / H_e * math.pi          # [H_e]
    # sin/cos grids
    sin_t = torch.sin(theta)           # [H_e]
    cos_t = torch.cos(theta)           # [H_e]
    # create per-pixel dirs: x=sinθ cosφ, y=cosθ, z=sinθ sinφ
    phi_g, th_g = torch.meshgrid(phi, theta, indexing='xy')
    sin_th = torch.sin(th_g);  cos_th = torch.cos(th_g)
    x = sin_th * torch.cos(phi_g)
    y = cos_th
    z = sin_th * torch.sin(phi_g)
    dirs = torch.stack((x,y,z), dim=-1).view(-1,3)  # [N,3], N=H_e*W_e

    # solid angle per pixel: dω = (2π/W_e)*(π/H_e)*sinθ
    dφ = 2 * math.pi / W_e
    dθ = math.pi / H_e
    # sinθ for each row j:
    sin_theta = sin_t  # [H_e]
    domega = (dφ * dθ * sin_theta).unsqueeze(1).expand(H_e, W_e)
    ω = domega.contiguous().view(-1)  # [N]

    # get radiance values per pixel, flatten
    L = env[0].permute(1,2,0).contiguous().view(-1,3)  # [N,3]

    # c0[channel]  = (1/(4π)) ∫ L dω
    integral0 = (L * ω.unsqueeze(1)).sum(dim=0)         # [3]
    c0 = integral0 / (4 * math.pi)

    # c1[channel] =        ∫ L(ω) * ω_dir  dω   -> shape [3,3]
    #   each channel has a 3-vector
    # matrix multiply: (L^T [3×N] @ (dirs * ω) [N×3])
    c1 = (L.t() @ (dirs * ω.unsqueeze(1)))              # [3,3]

    return c0, c1  # both torch tensors on device

# -- Disney BRDF (unchanged) --
def fresnel_schlick(cos_theta, F0):
    return F0 + (1 - F0) * (1 - cos_theta).clamp(0,1).pow(5)

def distribution_ggx(NdotH, alpha):
    a2 = alpha * alpha
    denom = ((NdotH * NdotH) * (a2 - 1) + 1).pow(2) * math.pi
    return a2 / denom.clamp(min=1e-8)

def geometry_schlick_ggx(NdotV, alpha):
    k = (alpha + 1).pow(2) / 8.0
    return NdotV / (NdotV * (1 - k) + k).clamp(min=1e-8)

def geometry_smith(NdotV, NdotL, alpha):
    return geometry_schlick_ggx(NdotV, alpha) * geometry_schlick_ggx(NdotL, alpha)

# -- Main renderer --
def render_disney_brdf(alb_img, norm_img, roug_img, metal_img, env_path, device=None):
    device = device or ('cuda' if torch.cuda.is_available() else 'cpu')

    # load material maps
    to_t = lambda img,mode: torch.from_numpy(np.array(img)).float().to(device).permute(1,2,0) if mode=='RGB' else torch.from_numpy(np.array(img)).float().to(device)[...,None]
    albedo = to_t(alb_img,'RGB')   / 255.0   # [H,W,3]
    normal = to_t(norm_img,'RGB')  / 255.0*2-1# [H,W,3]
    rough  = to_t(roug_img,'L')     / 255.0   # [H,W,1]
    metal  = to_t(metal_img,'L')    / 255.0   # [H,W,1]

    H,W = albedo.shape[:2]
    N = F.normalize(normal,dim=-1)           # [H,W,3]
    V = torch.tensor([0,0,1.],device=device).view(1,1,3).expand(H,W,3)

    # load environment map and precompute its SH coeffs
    env = load_exr_envmap(env_path, device)
    c0, c1 = compute_sh_coeffs(env)          # c0:[3], c1:[3,3]

    # --- DIFFUSE via SH ---
    # for each pixel: M = N @ c1^T   → [H*W,3]
    N_flat = N.view(-1,3)                    # [HW,3]
    M_flat = N_flat @ c1.t()                 # [HW,3]
    M = M_flat.view(H,W,3)                   # [H,W,3]

    # irradiance/π simplifies to (c0 + 2/3 * M)
    diffuse = albedo * ( c0.view(1,1,3) + (2/3) * M )

    # --- SPECULAR as before ---
    # reflection, GGX, etc.
    # sample env at R (single-sample mirror approx)
    # ... [you can reuse your previous sample_equirect implementation here] ...
    # For brevity we omit re‐pasting that code; it remains identical.

    # (copy your sample_equirect, specular computation, Fresnel, D, G here)

    # final = diffuse + specular
    # return final.clamp(0,1)

    # Placeholder to show the diffuse part works:
    return diffuse.clamp(0,1).cpu().numpy()

# -----------------------
# Example Usage
# -----------------------
if __name__ == "__main__":
    device = 'cuda' if torch.cuda.is_available() else 'cpu'
    # Load input images using imageio
    albedo = imageio.imread('albedo_000_000.png')
    normal = imageio.imread('normal_000_000.png') 
    roughness = imageio.imread('roughness.png')
    metallic = imageio.imread('metallic.png')
    
    # Render using Disney BRDF
    out = render_disney_brdf(albedo, normal, roughness, metallic, 'environment.exr', device)
    
    # Save output
    imageio.imwrite('render.png', (out * 255).astype(np.uint8))
