import logging
from typing import Literal, Optional
import os
import shutil

import torch
from PIL import Image

from diffusers import (
    CogVideoXDPMScheduler,
    CogVideoXImageToVideoPipeline,
    CogVideoXPipeline,
    CogVideoXVideoToVideoPipeline,
)
from diffusers.utils import export_to_video, load_image, load_video


logging.basicConfig(level=logging.INFO)

# Recommended resolution for each model (width, height)
RESOLUTION_MAP = {
    # cogvideox1.5-*
    "cogvideox1.5-5b-i2v": (768, 1360),
    "cogvideox1.5-5b": (768, 1360),
    # cogvideox-*
    "cogvideox-5b-i2v": (480, 720),
    "cogvideox-5b": (480, 720),
    "cogvideox-2b": (480, 720),
}


def generate_video(
    prompt: str,
    model_path: str,
    sft_path: str = None,
    lora_path: str = None,
    lora_rank: int = 128,
    num_frames: int = 81,
    width: Optional[int] = None,
    height: Optional[int] = None,
    output_path: str = "./output.mp4",
    image_or_video_path: str = "",
    num_inference_steps: int = 50,
    guidance_scale: float = 6.0,
    num_videos_per_prompt: int = 1,
    dtype: torch.dtype = torch.bfloat16,
    generate_type: str = Literal["t2v", "i2v", "v2v"],  # i2v: image to video, v2v: video to video
    seed: int = 42,
    fps: int = 16,
    mode: str = 'xyz',
):
    """
    Generates a video based on the given prompt and saves it to the specified path.

    Parameters:
    - prompt (str): The description of the video to be generated.
    - model_path (str): The path of the pre-trained model to be used.
    - lora_path (str): The path of the LoRA weights to be used.
    - lora_rank (int): The rank of the LoRA weights.
    - output_path (str): The path where the generated video will be saved.
    - num_inference_steps (int): Number of steps for the inference process. More steps can result in better quality.
    - num_frames (int): Number of frames to generate. CogVideoX1.0 generates 49 frames for 6 seconds at 8 fps, while CogVideoX1.5 produces either 81 or 161 frames, corresponding to 5 seconds or 10 seconds at 16 fps.
    - width (int): The width of the generated video, applicable only for CogVideoX1.5-5B-I2V
    - height (int): The height of the generated video, applicable only for CogVideoX1.5-5B-I2V
    - guidance_scale (float): The scale for classifier-free guidance. Higher values can lead to better alignment with the prompt.
    - num_videos_per_prompt (int): Number of videos to generate per prompt.
    - dtype (torch.dtype): The data type for computation (default is torch.bfloat16).
    - generate_type (str): The type of video generation (e.g., 't2v', 'i2v', 'v2v').·
    - seed (int): The seed for reproducibility.
    - fps (int): The frames per second for the generated video.
    """

    # 1.  Load the pre-trained CogVideoX pipeline with the specified precision (bfloat16).
    # add device_map="balanced" in the from_pretrained function and remove the enable_model_cpu_offload()
    # function to use Multi GPUs.

    image = None
    video = None

    model_name = model_path.split("/")[-1].lower()
    if model_name in RESOLUTION_MAP:
        desired_resolution = RESOLUTION_MAP[model_name]
    else:
        desired_resolution = (480, 720)
    if width is None or height is None:
        height, width = desired_resolution
        logging.info(f"\033[1mUsing default resolution {desired_resolution} for {model_name}\033[0m")
    elif (height, width) != desired_resolution:
        if generate_type == "i2v":
            # For i2v models, use user-defined width and height
            logging.warning(
                f"\033[1;31mThe width({width}) and height({height}) are not recommended for {model_name}. The best resolution is {desired_resolution}.\033[0m"
            )
        else:
            # Otherwise, use the recommended width and height
            logging.warning(
                f"\033[1;31m{model_name} is not supported for custom resolution. Setting back to default resolution {desired_resolution}.\033[0m"
            )
            height, width = desired_resolution
        
    if sft_path:
        print('loading SFT weight')
        if generate_type == "i2dpm":
            from core.finetune.models.custom_model import CogVideoXTransformer3DModelDomain
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelDomain.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-i2dpm":
            from core.finetune.models.custom_model import CogVideoXTransformer3DModelCondPM
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelCondPM.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-modtxt-i2dpm":
            from core.finetune.models.custom_model import CogVideoXTransformer3DModelCondPMModTxt
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelCondPMModTxt.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-modtxt-singledomain-i2dpm":
            from core.finetune.models.custom_model import CogVideoXTransformer3DModelCondPMModTxtSingleDomain
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelCondPMModTxtSingleDomain.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-singledomain-i2dpm":
            from core.finetune.models.custom_model import CogVideoXTransformer3DModelCondPMSingleDomain
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelCondPMSingleDomain.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-batchattn-singledomain-i2dpm":
            from core.finetune.models.custom_model_batch_attn import CogVideoXTransformer3DModelCondPMSingleDomainBatchAttn
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelCondPMSingleDomainBatchAttn.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "sbs-fullseq-i2dpm":
            from core.finetune.models.custom_model_sbs_fullseq import CogVideoXTransformer3DModelSBSFullSeq
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelSBSFullSeq.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "sbs-fullseq-diffrope-i2dpm":
            from core.finetune.models.custom_model_sbs_fullseq_diffrope import CogVideoXTransformer3DModelSBSFullSeqDiffRope
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelSBSFullSeqDiffRope.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "sbs-fullseq-clean-i2dpm":
            from core.finetune.models.custom_model_sbs_fullseq_clean import CogVideoXTransformer3DModelSBSFullSeqClean
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelSBSFullSeqClean.from_pretrained(sft_path, torch_dtype=dtype)
        elif generate_type == "condpm-dualproj-i2dpm":
            from core.finetune.models.custom_model_dual_proj import CogVideoXTransformer3DModelDualProj
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModelDualProj.from_pretrained(sft_path, torch_dtype=dtype)
        else:
            from diffusers import CogVideoXTransformer3DModel
            config_path = os.path.join(sft_path, 'config.json')
            src_config_path = os.path.join(model_path, 'transformer', 'config.json')
            if not os.path.exists(config_path):
                shutil.copyfile(src_config_path, config_path)
            transformer = CogVideoXTransformer3DModel.from_pretrained(sft_path, torch_dtype=dtype)
    else:
        transformer = None

    if generate_type == "i2v":
        pipe = CogVideoXImageToVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
        if mode == 'xyzrgb':
            pipe.vae.config._FrozenDict__frozen = False
            setattr(pipe.vae.config, 'invert_scale_latents', False)
            assert not pipe.vae.config.invert_scale_latents
            twidth, theight = image.size
            left_half = image.crop((0, 0, twidth // 2, theight))
            # Create a new image with the same size
            new_image = Image.new("RGB", (twidth, theight))
            # Paste the left half twice
            new_image.paste(left_half, (0, 0))
            new_image.paste(left_half, (twidth // 2, 0))
            image = new_image
    elif generate_type == "t2v":
        pipe = CogVideoXPipeline.from_pretrained(model_path, torch_dtype=dtype)
    elif generate_type == "i2dpm":
        from core.finetune.models.cogvideox_i2v.domain_sft_trainer import CogVideoXImageToDomainVideoPipeline
        pipe = CogVideoXImageToDomainVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpm_sft_trainer import CogVideoXImageToCondPMVideoPipeline
        pipe = CogVideoXImageToCondPMVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-modtxt-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpmmodtxt_sft_trainer import CogVideoXImageToCondPMModTxtVideoPipeline
        pipe = CogVideoXImageToCondPMModTxtVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-modtxt-singledomain-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpmmodtxt_sdomain_sft_trainer import CogVideoXImageToCondPMModTxtVideoPipeline
        pipe = CogVideoXImageToCondPMModTxtVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-singledomain-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpm_sdomain_sft_trainer import CogVideoXImageToCondPMVideoPipeline
        pipe = CogVideoXImageToCondPMVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-batchattn-singledomain-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpm_batchattn_sdomain_sft_trainer import CogVideoXImageToCondPMVideoPipeline
        pipe = CogVideoXImageToCondPMVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "sbs-fullseq-i2dpm":
        from core.finetune.models.cogvideox_i2v.sbs_fullseq_trainer import CogVideoXImageToSBSFullSeqVideoPipeline
        pipe = CogVideoXImageToSBSFullSeqVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    elif generate_type == "sbs-fullseq-diffrope-i2dpm":
        from core.finetune.models.cogvideox_i2v.sbs_fullseq_diffrope_trainer import CogVideoXImageToSBSFullSeqDiffRopeVideoPipeline
        pipe = CogVideoXImageToSBSFullSeqDiffRopeVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )   
        image = load_image(image=image_or_video_path)
    elif generate_type == "sbs-fullseq-clean-i2dpm":
        from core.finetune.models.cogvideox_i2v.sbs_fullseq_clean_trainer import CogVideoXImageToSBSFullSeqCleanVideoPipeline
        pipe = CogVideoXImageToSBSFullSeqCleanVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )   
        image = load_image(image=image_or_video_path)
    elif generate_type == "condpm-dualproj-i2dpm":
        from core.finetune.models.cogvideox_i2v.condpm_dual_proj_trainer import CogVideoXImageToCondPMDualProjVideoPipeline
        pipe = CogVideoXImageToCondPMDualProjVideoPipeline.from_pretrained(
            model_path, 
            transformer=transformer,
            torch_dtype=dtype
        )
        image = load_image(image=image_or_video_path)
    else:
        pipe = CogVideoXVideoToVideoPipeline.from_pretrained(model_path, torch_dtype=dtype)
        video = load_video(image_or_video_path)

        
    # If you're using with lora, add this code
    if lora_path:
        print('loading lora')
        pipe.load_lora_weights(lora_path, weight_name="pytorch_lora_weights.safetensors", adapter_name="cogvideox-lora")
        pipe.fuse_lora(components=["transformer"], lora_scale=64 / lora_rank)
        # pipe.set_adapters(["transformer"], [lora_scaling])

    # 2. Set Scheduler.
    # Can be changed to `CogVideoXDPMScheduler` or `CogVideoXDDIMScheduler`.
    # We recommend using `CogVideoXDDIMScheduler` for CogVideoX-2B.
    # using `CogVideoXDPMScheduler` for CogVideoX-5B / CogVideoX-5B-I2V.

    # pipe.scheduler = CogVideoXDDIMScheduler.from_config(pipe.scheduler.config, timestep_spacing="trailing")
    pipe.scheduler = CogVideoXDPMScheduler.from_config(pipe.scheduler.config, timestep_spacing="trailing")

    # 3. Enable CPU offload for the model.
    # turn off if you have multiple GPUs or enough GPU memory(such as H100) and it will cost less time in inference
    # and enable to("cuda")
    # pipe.to("cuda")

    # # pipe.enable_model_cpu_offload()
    # pipe.enable_sequential_cpu_offload()
    # pipe.vae.enable_slicing()
    # pipe.vae.enable_tiling()

    pipe.to("cuda")

    # 4. Generate the video frames based on the prompt.
    if generate_type == "i2v":
        video_generate = pipe(
            height=height,
            width=width,
            prompt=prompt,
            image=image,
            # The path of the image, the resolution of video will be the same as the image for CogVideoX1.5-5B-I2V, otherwise it will be 720 * 480
            num_videos_per_prompt=num_videos_per_prompt,  # Number of videos to generate per prompt
            num_inference_steps=num_inference_steps,  # Number of inference steps
            num_frames=num_frames,  # Number of frames to generate
            use_dynamic_cfg=True,  # This id used for DPM scheduler, for DDIM scheduler, it should be False
            guidance_scale=guidance_scale,
            generator=torch.Generator().manual_seed(seed),  # Set the seed for reproducibility
            output_type="latent"
        ).frames[0]
    elif generate_type == "t2v":
        video_generate = pipe(
            height=height,
            width=width,
            prompt=prompt,
            num_videos_per_prompt=num_videos_per_prompt,
            num_inference_steps=num_inference_steps,
            num_frames=num_frames,
            use_dynamic_cfg=True,
            guidance_scale=guidance_scale,
            generator=torch.Generator().manual_seed(seed),
            output_type="latent"
        ).frames[0]
    elif generate_type == "i2dpm" or generate_type == "condpm-i2dpm" or generate_type == "condpm-modtxt-i2dpm" or generate_type == "condpm-modtxt-singledomain-i2dpm" or generate_type == "condpm-singledomain-i2dpm" or generate_type == "condpm-batchattn-singledomain-i2dpm" or generate_type == "sbs-fullseq-i2dpm" or generate_type == "sbs-fullseq-diffrope-i2dpm" or generate_type == "sbs-fullseq-clean-i2dpm" or generate_type == "condpm-dualproj-i2dpm":
        video_generate = pipe(
            height=height,
            width=width,
            prompt=prompt,
            image=image,
            # The path of the image, the resolution of video will be the same as the image for CogVideoX1.5-5B-I2V, otherwise it will be 720 * 480
            num_videos_per_prompt=num_videos_per_prompt,  # Number of videos to generate per prompt
            num_inference_steps=num_inference_steps,  # Number of inference steps
            num_frames=num_frames,  # Number of frames to generate
            use_dynamic_cfg=True,  # This id used for DPM scheduler, for DDIM scheduler, it should be False
            guidance_scale=guidance_scale,
            generator=torch.Generator().manual_seed(seed),  # Set the seed for reproducibility
            output_type="latent"
        ).frames
    else:
        raise NotImplementedError

    return video_generate