import os
import sys
import json
import shutil
from safetensors.torch import load_file, save_file
import torch
import argparse

# Usage: python get_emb_from_ckpt.py <checkpoint-xxxx path>

def main(ckpt_path):
    # 1. Get the converted_checkpoint path
    ckpt_dir = os.path.abspath(ckpt_path)
    base_dir, ckpt_name = os.path.split(ckpt_dir)
    if not ckpt_name.startswith('checkpoint-'):
        raise ValueError('Checkpoint folder should start with checkpoint-')
    step = ckpt_name.split('-')[-1]
    converted_dir = os.path.join(base_dir, f'converted-{step}')
    converted_lite_dir = os.path.join(base_dir, f'converted-lite-{step}')
    os.makedirs(converted_lite_dir, exist_ok=True)

    # 2. Read the index file
    index_path = os.path.join(converted_dir, 'diffusion_pytorch_model.safetensors.index.json')
    with open(index_path, 'r') as f:
        index_data = json.load(f)
    weight_map = index_data.get('weight_map', {})
    if 'learnable_domain_embeddings' not in weight_map:
        raise KeyError('learnable_domain_embeddings not found in index file')
    shard_name = weight_map['learnable_domain_embeddings']
    shard_path = os.path.join(converted_dir, shard_name)

    # 3. Load the specific shard
    shard_weights = load_file(shard_path)
    if 'learnable_domain_embeddings' not in shard_weights:
        raise KeyError('learnable_domain_embeddings not found in the shard')
    emb = shard_weights['learnable_domain_embeddings']

    # 4. Save the learnable_domain_embeddings to a file in converted-lite-xxxx
    emb_out_path = os.path.join(converted_lite_dir, 'learnable_domain_embeddings.pt')
    torch.save(emb, emb_out_path)
    print(f'Saved learnable_domain_embeddings to {emb_out_path}')

    # 5. Move pytorch_lora_weights.safetensors to converted-lite-xxxx
    lora_path = os.path.join(ckpt_dir, 'pytorch_lora_weights.safetensors')
    if os.path.exists(lora_path):
        shutil.copy2(lora_path, os.path.join(converted_lite_dir, 'pytorch_lora_weights.safetensors'))
        print(f'Moved pytorch_lora_weights.safetensors to {converted_lite_dir}')
    else:
        print(f'Warning: {lora_path} does not exist.')

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Extract learnable_domain_embeddings and move lora weights.')
    parser.add_argument('checkpoint_path', type=str, help='Path to the checkpoint-xxxx directory')
    args = parser.parse_args()
    main(args.checkpoint_path)