import glob
import time
import pickle
import itertools
from pathlib import Path

import numpy as np
import tyro
from tqdm.auto import tqdm

import rerun as rr
import imageio

from core.dataset import PointmapDataset
from core.annotation import Monst3RAnno

import torch
import torch.nn.functional as F
# ---------- helpers -----------------------------------------------------------
def rot6d_to_matrix(d6: torch.Tensor) -> torch.Tensor:
    """
    Convert the 6-D rotation representation to a 3×3 matrix.
    (Zhou et al., ‘On the Continuity of Rotation Representations…’, CVPR’19)
    d6: (6,)  tensor – returns (3,3)
    """
    a1, a2 = d6[:3], d6[3:]
    b1 = F.normalize(a1, dim=0)
    b2 = F.normalize(a2 - (a2 @ b1) * b1, dim=0)
    b3 = torch.cross(b1, b2)
    return torch.stack([b1, b2, b3], dim=1)   # 3×3


def build_K(fx, fy, cx, cy) -> torch.Tensor:
    """Construct the 3×3 intrinsics matrix"""
    return torch.tensor([[fx, 0., cx],
                         [0., fy, cy],
                         [0., 0., 1.]], device=fx.device)


# ---------- main optimisation -------------------------------------------------
def optimise_depth_and_camera(pred_xyz: torch.Tensor,
                              n_iters: int = 2000,
                              lr: float = 1e-2,
                              verbose: int = 100):
    """
    Args
    ----
    pred_xyz : (H,W,3) tensor with the network-predicted XYZ image  x̂^{XYZ}
    Returns
    -------
    depth     : (H,W)   refined depth
    K         : (3,3)   refined intrinsics
    R, t      : (3,3), (3,)  refined extrinsics  (camera→world)
    """

    device = pred_xyz.device
    H, W, _ = pred_xyz.shape
    N       = H * W

    # ----- pixel grid ---------------------------------------------------------
    ys, xs = torch.meshgrid(torch.arange(H, device=device),
                            torch.arange(W, device=device),
                            indexing='ij')
    pix_h  = torch.stack([xs, ys, torch.ones_like(xs)], dim=-1)       # (H,W,3)
    pix_h  = pix_h.reshape(-1, 3).T                                   # (3,N)

    # ----- parameters to optimise -------------------------------------------
    log_depth = torch.nn.Parameter(torch.log(pred_xyz.norm(dim=2) + 1e-3))   # (H,W)
    fx = torch.nn.Parameter(torch.tensor(W * 1.2, device=device))
    fy = torch.nn.Parameter(torch.tensor(H * 1.2, device=device))
    cx = torch.nn.Parameter(torch.tensor(W / 2,  device=device))
    cy = torch.nn.Parameter(torch.tensor(H / 2,  device=device))
    rot6d = torch.nn.Parameter(torch.eye(3, device=device)[:, :2].reshape(-1))  # (6,)
    trans = torch.nn.Parameter(torch.zeros(3, device=device))

    optim = torch.optim.Adam([log_depth, fx, fy, cx, cy, rot6d, trans], lr=lr)
    target = pred_xyz.reshape(-1, 3)                                           # (N,3)

    for it in range(1, n_iters + 1):
        optim.zero_grad()

        # --- unpack parameters ---------------------------------------------
        depth = F.softplus(log_depth).reshape(-1, 1)          # (N,1)  keep depth > 0
        K     = build_K(fx, fy, cx, cy)
        Kinv  = torch.inverse(K)

        R = rot6d_to_matrix(rot6d)
        P_cam2world = torch.eye(4, device=device)
        P_cam2world[:3, :3] = R
        P_cam2world[:3,  3] = trans
        # -------------------------------------------------------------------

        cam_pts   = Kinv @ (depth * pix_h)                    # (3,N)
        cam_pts_h = torch.cat([cam_pts, torch.ones(1, N, device=device)], 0)  # (4,N)
        world_pts = (P_cam2world @ cam_pts_h)[:3].T           # (N,3)

        loss = (world_pts - target).pow(2).mean()
        loss.backward()
        optim.step()

        if verbose and it % verbose == 0:
            print(f"[{it:4d}/{n_iters}]  reprojection-loss = {loss.item():.6f}")

    depth_map = F.softplus(log_depth).detach()
    return depth_map, K.detach(), R.detach(), trans.detach()


def log_image_plane_outline(rr, name, K, T_world_camera, H, W):
    z = 1.0
    corners_px = np.array([
        [0, 0],
        [W, 0],
        [W, H],
        [0, H],
        [0, 0],  # Close the loop
    ])
    Kinv = np.linalg.inv(K)
    corners_cam = []
    for u, v in corners_px:
        vec = np.array([u, v, 1.0])
        xyz = Kinv @ vec
        xyz = xyz / xyz[2] * z
        corners_cam.append(xyz)
    corners_cam = np.stack(corners_cam, axis=0)
    R = T_world_camera[:3, :3]
    t = T_world_camera[:3, 3]
    corners_world = (R @ corners_cam.T).T + t
    rr.log(
        f"{name}/image_frustum_outline",
        rr.LineStrips3D([corners_world])
    )

def main(
    downsample_factor: int = 1,
    frame_gap: int = 1,
    max_frames: int = 100,
    use_mask: bool = False,
    pkl_dir: str = None,
    monst3r_dir: str = None,
    pointmap_mp4: str = None,
    rgb_mp4: str = None,
    pointmap_npy: str = None,
    rr_recording: str = "pointmap_log",
) -> None:
    rr.init(rr_recording)

    # Data loading logic (same as vis.py, with new mp4 support)
    if pointmap_npy is not None and rgb_mp4 is not None:
        pointmap = np.load(pointmap_npy)  # [num_frames, H, W, 3]
        rgb_reader = imageio.get_reader(rgb_mp4, "ffmpeg")
        num_frames = min(pointmap.shape[0], len(rgb_reader), max_frames)
        def PointmapIterator():
            for i in range(num_frames):
                pm = pointmap[i]  # H, W, 3
                rgb = rgb_reader.get_data(i).astype(np.float32) / 255.0  # H, W, 3
                H, W, _ = pm.shape
                yield type('Frame', (), {
                    'pcd': pm.reshape(-1, 3),
                    'pcd_color': rgb.reshape(-1, 3),
                    'rgb': rgb,
                    'T_world_camera': np.eye(4),
                }), f"npy_frame"
    elif pointmap_mp4 is not None and rgb_mp4 is not None:
        pointmap_reader = imageio.get_reader(pointmap_mp4, "ffmpeg")
        rgb_reader = imageio.get_reader(rgb_mp4, "ffmpeg")
        num_frames = 81
        def PointmapIterator():
            for i in range(num_frames):
                pm = pointmap_reader.get_data(i).astype(np.float32) / 255.0  # H, W, 3
                rgb = rgb_reader.get_data(i).astype(np.float32) / 255.0  # H, W, 3
                H, W, _ = pm.shape
                # Flatten for rerun
                yield type('Frame', (), {
                    'pcd': pm.reshape(-1, 3),
                    'pcd_color': rgb.reshape(-1, 3),
                    'rgb': rgb,
                    'T_world_camera': np.eye(4),  # Identity if unknown
                }), f"mp4_frame"
    elif pkl_dir is not None:
        pkl_list = glob.glob(f'{pkl_dir}/*.pkl')
        def PointmapIterator():
            for pkl_path in sorted(pkl_list):
                yield pickle.load(open(pkl_path, 'rb')), pkl_path
    elif monst3r_dir is not None:
        monst3r_list = glob.glob(f'{monst3r_dir}/*/')
        def PointmapIterator():
            for scene_path in monst3r_list:
                yield Monst3RAnno(scene_path, max_frames=max_frames).pointmap, scene_path
    else:
        print("Loading frames!")
        dataset = PointmapDataset(datalist = 'data/sanity_check.txt', debug=False, random_shuffle=True, max_frames=max_frames, skip_invalid=False)
        def PointmapIterator():
            for data in dataset:
                yield data.pointmap, data.anno_dir.split('s3://')[-1]

    pointmap_iterator = PointmapIterator()
    # Before logging, I want to optimize the pointmap 

    
    if rgb_mp4 is not None:
        # Each iteration yields a single frame and name, so just log it once
        idx = 0
        for (frame, name) in pointmap_iterator:
            position, color = frame.pcd, frame.pcd_color
            rgb = frame.rgb
            H, W, _ = rgb.shape
            position = position.reshape(H, W, 3)[::downsample_factor, ::downsample_factor].reshape([-1, 3])
            color = color.reshape(H, W, 3)[::downsample_factor, ::downsample_factor].reshape([-1, 3])
            rr.set_time_sequence("frame", idx)  # Use frame index from name
            # Log point cloud
            rr.log(
                f"{name}/point_cloud",
                rr.Points3D(
                    positions=position,
                    colors=(color * 255).astype(np.uint8),
                ),
            )
            # Log RGB image (optional)
            rr.log(
                f"{name}/rgb_image",
                rr.Image(rgb),
            )
            # Log camera frustum (if possible)
            # rerun does not have a direct frustum primitive, but we can log camera pose as transform
            rr.log(
                f"{name}/camera",
                rr.Transform3D(
                    translation=frame.T_world_camera[:3, 3],
                    mat3x3=frame.T_world_camera[:3, :3],
                ),
            )
            if use_mask and hasattr(frame, 'fg_pcd'):
                # Foreground
                rr.log(
                    f"{name}/fg_point_cloud",
                    rr.Points3D(
                        positions=frame.fg_pcd,
                        colors=(frame.fg_pcd_color * 255).astype(np.uint8),
                    ),
                )
                # Background
                rr.log(
                    f"{name}/bg_point_cloud",
                    rr.Points3D(
                        positions=frame.bg_pcd,
                        colors=(frame.bg_pcd_color * 255).astype(np.uint8),
                    ),
                )
            idx += 1
    else:
        for (loader, name) in pointmap_iterator:
            num_frames = min(max_frames, loader.num_frames())
            for i in tqdm(range(0, num_frames, frame_gap)):
                frame = loader.get_frame(i)
                position, color = frame.pcd, frame.pcd_color
                rgb = frame.rgb
                H, W, _ = rgb.shape
                position = position.reshape(H, W, 3)[::downsample_factor, ::downsample_factor].reshape([-1, 3])
                color = color.reshape(H, W, 3)[::downsample_factor, ::downsample_factor].reshape([-1, 3])
                rr.set_time_sequence("frame", i)
                # Log point cloud
                rr.log(
                    f"{name}/point_cloud",
                    rr.Points3D(
                        positions=position,
                        colors=(color * 255).astype(np.uint8),
                    ),
                )
                # Log RGB image (optional)
                rr.log(
                    f"{name}/rgb_image",
                    rr.Image(rgb),
                )
                # Log camera frustum (if possible)
                # rerun does not have a direct frustum primitive, but we can log camera pose as transform
                rr.log(
                    f"{name}/camera",
                    rr.Transform3D(
                        translation=frame.T_world_camera[:3, 3],
                        mat3x3=frame.T_world_camera[:3, :3],
                    ),
                )

                # log_image_plane_outline(rr, name, frame.K, frame.T_world_camera, H, W)

                if use_mask and hasattr(frame, 'fg_pcd'):
                    # Foreground
                    rr.log(
                        f"{name}/fg_point_cloud",
                        rr.Points3D(
                            positions=frame.fg_pcd,
                            colors=(frame.fg_pcd_color * 255).astype(np.uint8),
                        ),
                    )
                    # Background
                    rr.log(
                        f"{name}/bg_point_cloud",
                        rr.Points3D(
                            positions=frame.bg_pcd,
                            colors=(frame.bg_pcd_color * 255).astype(np.uint8),
                        ),
                    )
            print(f"Finished logging {name}. Open rerun viewer to inspect.")
            # break
    rr.save(rr_recording)

if __name__ == "__main__":
    tyro.cli(main)
