import sys
from pathlib import Path
sys.path.append(str(Path(__file__).parent.parent))

import imageio
import numpy as np
import glob
import os
from core.dataset import PointmapDataset
from core.annotation import Monst3RAnno
import pickle

from core.tokenizer.cogvideo import CogvideoTokenizer
from core.tokenizer.wan import WanTokenizer


def transform_pointclouds_to_first_camera(pointclouds, cams2world):
    """
    pointclouds: [F, N, 3] in world coordinates
    cams2world: [F, 4, 4] original camera-to-world matrices
    Returns: [F, N, 3] pointclouds in the new coordinate system (first camera as origin)
    """
    # Invert the first pose
    world2first = np.linalg.inv(cams2world[0])  # [4, 4]
    
    # Convert pointclouds to homogeneous coordinates
    F, N, _ = pointclouds.shape
    ones = np.ones((F, N, 1), dtype=pointclouds.dtype)
    points_homo = np.concatenate([pointclouds, ones], axis=-1)  # [F, N, 4]
    
    # Apply the transformation
    # (world2first @ point_homo.T).T for each frame
    points_transformed = np.einsum('ij,fkj->fki', world2first, points_homo)  # [F, N, 4]
    
    # Drop the homogeneous coordinate
    return points_transformed[..., :3]

max_frames = 29

dataset = PointmapDataset(datalist = 'data/sanity_check.txt', s3_conf_path='~/petreloss.conf', debug=True, random_shuffle=False, max_frames=max_frames)
def PointmapIterator():
    for data in dataset:
        yield data.pointmap

pointmap_iterator = PointmapIterator()

model_path = 'pretrained/Wan2.1-I2V-14B-480P-Diffusers/vae/'
tokenizer = WanTokenizer(model_path=model_path)
# model_path = 'pretrained/CogVideoX1.5-5B-I2V/vae/'
# tokenizer = CogvideoTokenizer(model_path=model_path)

save_dir = 'results/tokenizer_pointmap_test'
os.makedirs(save_dir, exist_ok=True)
for pm in pointmap_iterator:
    pickle.dump(pm, open(f'{save_dir}/pm.pkl', 'wb'))
    pointmap = pm.pcd
    print('pointmap range: ', pointmap.min(), pointmap.max())
    pointmap = np.clip(pointmap, np.percentile(pointmap, 2, axis=1, keepdims=True), np.percentile(pointmap, 98,  axis=1, keepdims=True))
    print('clipped range: ', pointmap.min(), pointmap.max())
    pointmap_in, center, scale = tokenizer.normalize_instance(pointmap, range_min=0)
    pointmap_in = pointmap_in.reshape(*pm.rgb.shape)
    pointmap_vis = (pointmap_in * 255).astype(np.uint8)

    latents, aux = tokenizer.encode(pointmap_in)
    pointmap_recon = tokenizer.decode(latents, **aux)
    print(pointmap_recon.shape)

    pointmap_recon_vis = (np.clip(pointmap_recon, 0, 1) * 255).astype(np.uint8)
    imageio.mimwrite(f'{save_dir}/pointmap_comp.mp4', np.concatenate([pointmap_vis, pointmap_recon_vis], -2), fps=24)

    pointmap_recon = pointmap_recon.reshape(*pm.pcd.shape)
    pm.pcd = pointmap_recon
    pickle.dump(pm, open(f'{save_dir}/pm_recon.pkl', 'wb'))
    pointmap_recon = tokenizer.denormalize_instance(pointmap_recon, center, scale, range_min=0)
    print('reconstructed range: ', pointmap_recon.min(), pointmap_recon.max())
    pm.pcd = pointmap_recon
    pickle.dump(pm, open(f'{save_dir}/pm_recon_gtscale.pkl', 'wb'))
    align_first_pm = transform_pointclouds_to_first_camera(pointmap, pm.cams2world)
    pm.pcd = align_first_pm
    pickle.dump(pm, open(f'{save_dir}/pm_align.pkl', 'wb'))
    break
