import os
import numpy as np
import imageio
import torch
import torchvision.transforms.functional as F
import torch.nn.functional as nnf
import argparse
from core.dataset import PointmapDataset
from core.tokenizer.cogvideo import CogvideoTokenizer


def split_list(datalist, num_segments):
    """Splits list datalist into num_segments continuous segments with balanced load."""
    n = len(datalist)
    segment_size = n // num_segments
    remainder = n % num_segments  # Extra elements to distribute

    segments = []
    start = 0
    for i in range(num_segments):
        extra = 1 if i < remainder else 0  # Distribute remainder among the first few segments
        end = start + segment_size + extra
        segments.append(datalist[start:end])
        start = end

    return segments


def main(args):
    data_dir = args.out
    os.makedirs(data_dir, exist_ok=True)
    os.makedirs(f'{data_dir}/videos', exist_ok=True)
    os.makedirs(f'{data_dir}/video_latents', exist_ok=True)
    os.makedirs(f'{data_dir}/pointmap', exist_ok=True)
    os.makedirs(f'{data_dir}/pointmap_latents', exist_ok=True)
    os.makedirs(f'{data_dir}/datalist', exist_ok=True)
    os.makedirs(f'{data_dir}/prompts', exist_ok=True)
    os.makedirs(f'{data_dir}/center', exist_ok=True)
    os.makedirs(f'{data_dir}/scale', exist_ok=True)

    max_frames = 81     # CogVideo 1.5 required framelength
    resolution_out = (768, 1360)     # CogVideo 1.5 required resolution
    resolution_crop = resolution_out

    pexel_datalist = []
    with open('data/filtered_pexelx.txt', 'r') as f:
        for line in f.readlines():
            pexel_datalist.append(line.strip())
    pexel_datalist_segment = split_list(pexel_datalist, num_segments=args.num_tasks)[args.task_idx]
    print(f'{args.task_idx}/{args.num_tasks} task | {len(pexel_datalist_segment)}/{len(pexel_datalist)} samples')
    # dataset
    cache_dir = f'.cache/{args.task_idx:05d}/'
    train_dataset_iterator = iter(
            PointmapDataset(
                datalist = pexel_datalist_segment,
                max_frames = max_frames,
                s3_conf_path = '~/petreloss.conf',
                debug=False,
                random_shuffle=False,
                cache_dir=cache_dir,
                skip_invalid=False
        )
    )

    # tokenizer
    model_path = 'pretrained/CogVideoX1.5-5B-I2V/vae/'
    # we do not apply scaling_factor when caching the latents of pointmap
    tokenizer = CogvideoTokenizer(model_path=model_path)

    prompt_list = []
    data_list = []
    center_list = []
    scale_list = []
    for i in range(len(pexel_datalist_segment)):
        data = next(train_dataset_iterator)
        if data is None:
            continue
        pm_resolution = data.pointmap.get_resolution()
        pm_num_frames = data.pointmap.num_frames()

        caption = data.video.caption
        prompt_list.append(caption)
        video_name = os.path.basename(data.video.path).replace('.mp4', '')
        if data.clip_start > -1:
            video_name = video_name + f'-{data.clip_start:04d}-{data.clip_start+data.length:04d}'
        video_path = f'{data_dir}/videos/{video_name}.mp4'
        data_list.append(f'videos/{video_name}.mp4')
        
        # save rgb
        rgb = data.rgb_raw
        # Get current dimensions and target dimensions
        T, H, W, C = rgb.shape
        target_h, target_w = resolution_out
        
        # Calculate resize dimensions to match shortest side while maintaining aspect ratio
        if H/W < target_h/target_w:  # Height is shorter side
            new_h = target_h
            new_w = int(W * (target_h/H))
        else:  # Width is shorter side
            new_w = target_w 
            new_h = int(H * (target_w/W))
            
        # Resize rgb to match shortest side
        rgb = nnf.interpolate(torch.from_numpy(rgb.transpose(0,3,1,2)), 
                            size=(new_h, new_w), 
                            mode='bilinear', 
                            align_corners=False,
                            antialias=True).numpy().transpose(0,2,3,1)
        rgb = F.center_crop(torch.from_numpy(rgb.transpose(0,3,1,2)), resolution_out).numpy().transpose(0,2,3,1)
        imageio.mimwrite(video_path, (rgb * 255).clip(0, 255).astype(np.uint8), fps=24)
        
        # save pointmap latents
        pointmap_latent_path = f'{data_dir}/pointmap_latents/{video_name}.pt'
        pointmap = data.pointmap.pcd
        pointmap = np.clip(pointmap, np.percentile(pointmap, 2, axis=1, keepdims=True), np.percentile(pointmap, 98,  axis=1, keepdims=True))
        pointmap_in, center, scale = tokenizer.normalize_instance(pointmap, range_min=0)
        center_list.append(center)
        scale_list.append(scale)
        pointmap_in = pointmap_in.reshape(*data.pointmap.rgb.shape)
        pm_path = f'{data_dir}/pointmap/{video_name}.mp4'
        imageio.mimwrite(pm_path, (pointmap_in * 255).clip(0, 255).astype(np.uint8), fps=24)
        pointmap_in = nnf.interpolate(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), 
                            size=(new_h, new_w), 
                            mode='nearest').numpy().transpose(0,2,3,1)
        pointmap_in = F.center_crop(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), resolution_crop).numpy().transpose(0,2,3,1)
        torch.save(tokenizer.encode(pointmap_in)[0], pointmap_latent_path)
        import pdb; pdb.set_trace()
        
        print(f'Finish {i}.')

    # backup segment meta data
    with open(f'{data_dir}/prompts/{args.task_idx:05d}.txt', 'w') as f:
        f.write('\n'.join(prompt_list))
    with open(f'{data_dir}/datalist/{args.task_idx:05d}.txt', 'w') as f:
        f.write('\n'.join(data_list))

    center_path = f'{data_dir}/center/{args.task_idx:05d}.npy'
    scale_path = f'{data_dir}/scale/{args.task_idx:05d}.npy'
    np.save(center_path, np.stack(center_list))
    np.save(scale_path, np.stack(scale_list))

    print(f'All finished.')

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='build pointmap dataset')
    parser.add_argument('--out', type=str, default='data/feb23_debug')
    parser.add_argument('--num_tasks', type=int, default=1000)
    parser.add_argument('--task_idx', type=int, default=0)
    args = parser.parse_args()
    
    main(args)
    
