        # # 1. Patch & position embedding
        # self.rope = WanRotaryPosEmbed(attention_head_dim, patch_size, rope_max_seq_len)
        # self.patch_embedding = nn.Conv3d(in_channels, inner_dim, kernel_size=patch_size, stride=patch_size)
        
        # batch_size, num_channels, num_frames, height, width = hidden_states.shape
        # p_t, p_h, p_w = self.config.patch_size
        # post_patch_num_frames = num_frames // p_t
        # post_patch_height = height // p_h
        # post_patch_width = width // p_w

        # rotary_emb = self.rope(hidden_states)

        # hidden_states = self.patch_embedding(hidden_states)
        # hidden_states = hidden_states.flatten(2).transpose(1, 2)

import torch
import torch.nn as nn
import imageio.v2 as imageio
import os
import numpy as np

def save_token_grid(tokens, seq_len, inner_dim, prefix, out_dir="patchify_visuals"):
    os.makedirs(out_dir, exist_ok=True)
    # tokens: (batch, seq_len//2, inner_dim)
    # For visualization, reduce inner_dim to 1 (mean) and reshape to a grid
    grid_size = int(seq_len ** 0.5)
    if grid_size * grid_size != seq_len:
        grid_size = int((seq_len//2) ** 0.5)
    # Take mean over inner_dim for visualization
    vis = tokens[0].mean(dim=-1).detach().cpu().numpy()
    # Pad to square if needed
    pad_len = grid_size * grid_size - vis.shape[0]
    if pad_len > 0:
        vis = np.pad(vis, (0, pad_len), mode='constant')
    vis = vis.reshape(grid_size, grid_size)
    # Normalize to 0-255 for uint8 image
    vis_min, vis_max = vis.min(), vis.max()
    if vis_max > vis_min:
        vis = (vis - vis_min) / (vis_max - vis_min)
    vis_img = (vis * 255).astype('uint8')
    imageio.imwrite(os.path.join(out_dir, f"{prefix}.png"), vis_img)

def save_token_video(tokens, num_frames, grid_h, grid_w, prefix, out_dir="patchify_visuals"):
    os.makedirs(out_dir, exist_ok=True)
    # tokens: (batch, seq_len, inner_dim)
    # Reshape to (batch, num_frames, grid_h, grid_w, inner_dim)
    batch = tokens.shape[0]
    inner_dim = tokens.shape[-1]
    seq_len = tokens.shape[1]
    assert batch == 1, "Only batch size 1 supported for visualization."
    tokens = tokens[0]  # (seq_len, inner_dim)
    tokens = tokens.view(num_frames, grid_h * grid_w, inner_dim)
    frames = []
    for t in range(num_frames):
        vis = tokens[t].mean(-1).detach().cpu().numpy()
        # Pad if needed
        pad_len = grid_h * grid_w - vis.shape[0]
        if pad_len > 0:
            vis = np.pad(vis, (0, pad_len), mode='constant')
        vis = vis.reshape(grid_h, grid_w)
        vis_min, vis_max = vis.min(), vis.max()
        if vis_max > vis_min:
            vis = (vis - vis_min) / (vis_max - vis_min)
        vis_img = (vis * 255).astype('uint8')
        frames.append(vis_img)
    video_path = os.path.join(out_dir, f"{prefix}.mp4")
    imageio.mimwrite(video_path, frames, fps=4)

def test_patchify_token_order_visual():
    batch_size = 1
    channels = 3
    frames = 16
    height = 64
    width = 64
    patch_size = (1, 1, 1)  # typical video patch size
    inner_dim = 3

    # Calculate patch grid
    p_t, p_h, p_w = patch_size
    num_frames = frames // p_t
    grid_h = height // p_h
    grid_w = width // p_w
    num_patches_per_frame = grid_h * grid_w
    total_patches = num_frames * num_patches_per_frame

    # Create two different tensors with distinct patterns
    tensor1 = torch.zeros(batch_size, channels, frames//2, height, width)
    tensor2 = torch.ones(batch_size, channels, frames//2, height, width)
    
    # Test frame dimension concatenation
    frame_concat = torch.cat([tensor1, tensor2], dim=2)
    patch_embed = nn.Identity()
    
    # Patchify
    patched = patch_embed(frame_concat)
    tokens = patched.flatten(2).transpose(1, 2)
    
    # Split sequence back
    seq_len = tokens.shape[1]
    first_half, second_half = tokens.split(seq_len//2, dim=1)
    
    # Visualize as video
    save_token_video(first_half, num_frames//2, grid_h, grid_w, "frame_first_half")
    save_token_video(second_half, num_frames//2, grid_h, grid_w, "frame_second_half")

    # Test height dimension
    tensor1 = torch.zeros(batch_size, channels, frames, height//2, width)
    tensor2 = torch.ones(batch_size, channels, frames, height//2, width)
    height_concat = torch.cat([tensor1, tensor2], dim=3)
    
    patched = patch_embed(height_concat)
    tokens = patched.flatten(2).transpose(1, 2)
    first_half, second_half = tokens.split(seq_len//2, dim=1)
    save_token_video(first_half, num_frames, grid_h//2, grid_w, "height_first_half")
    save_token_video(second_half, num_frames, grid_h//2, grid_w, "height_second_half")

    # Test width dimension
    tensor1 = torch.zeros(batch_size, channels, frames, height, width//2)
    tensor2 = torch.ones(batch_size, channels, frames, height, width//2)
    width_concat = torch.cat([tensor1, tensor2], dim=4)
    
    patched = patch_embed(width_concat)
    tokens = patched.flatten(2).transpose(1, 2)
    first_half, second_half = tokens.split(seq_len//2, dim=1)
    save_token_video(first_half, num_frames, grid_h, grid_w//2, "width_first_half")
    save_token_video(second_half, num_frames, grid_h, grid_w//2, "width_second_half")

if __name__ == "__main__":
    import numpy as np
    test_patchify_token_order_visual()